/*******************************************************************************
 mp4_process.h -

 Copyright (C) 2007-2025 CodeShop B.V.
 http://www.code-shop.com

 For licensing see the LICENSE file
******************************************************************************/

#ifndef MP4_PROCESS_H_AKW
#define MP4_PROCESS_H_AKW

#include "mod_streaming_export.h"
#include "mp4_error.h"
#include "mp4_log.h"
#include <stddef.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct mp4_global_context_t mp4_global_context_t;
typedef struct mp4_process_context_t mp4_process_context_t;
typedef struct buckets_t buckets_t;
typedef struct headers_t headers_t;
typedef struct mp4_options_t mp4_options_t;

// Following section carries a global init/exit function which should be
// called as first and as last function of the API (before and after
// anything else)

MP4_DLL_EXPORT
mp4_global_context_t* libfmp4_global_init();

MP4_DLL_EXPORT
void libfmp4_global_exit(mp4_global_context_t const* global_context);

MP4_DLL_EXPORT
char const* libfmp4_load_license(mp4_global_context_t* global_context,
                                 char const* src,
                                 char const* version,
                                 char const* license);

MP4_DLL_EXPORT
char const* libfmp4_print_license(mp4_global_context_t* global_context,
                                  char const* src,
                                  char const* version,
                                  char const* license);

// Function type for implementing a download handler in a webserver module.
// Returns the HTTP status (200, etc) if the GET request succeeds.
// Returns AP_FILTER_ERROR when the download handler failed internally.
typedef int (*download_callback_t)(void* context,
                                   char const* url,
                                   uint64_t* offset,
                                   uint32_t* len,
                                   uint64_t* size,
                                   headers_t* headers,
                                   buckets_t* buckets);

// Requests the server variable 'name'. The pointer to the value must be
// returned in *value.  Returns the size of the string.
// E.g. "REQUEST_SCHEME", "HTTP_HOST", "REQUEST_URI"

typedef int (*get_server_variable_callback_t)(void* context,
                                              char const* name,
                                              char const** value);

typedef enum
{
  FMP4_HTTP_GET = 0, // read
  FMP4_HTTP_PUT,     // update
  FMP4_HTTP_POST,    // create
  FMP4_HTTP_DELETE   // delete
} fmp4_http_method_t;

// The mp4_process function is the main entry point for the libfmp4 library.
//
// An example for retrieving the client manifest file would be as follows:
//
// Set the filename/url to process
// char const* filename = "/var/www/videos/video.ism";
//
// Set the vpath
// char const* args = "file=Manifest";
//
// Create and initialize the context structure.
// mp4_process_context_t* context = mp4_process_context_init(global_context);
//
// Parse the options
// mp4_options_t* options = mp4_process_context_get_options(context);
// mp4_options_set(options, args, strlen(args));
//
// Call the library
// int http_status = mp4_process(context, FMP4_HTTP_GET, filename);
//
// Check for an error
// if(http_status != 200)
// {
//   error(mp4_process_context_get_result_text(context));
// }
//
// The output is stored in the buckets.
//
// buckets_t* buckets = mp4_process_context_get_buckets(context);
// bucket_t* bucket = buckets->bucket_;
// if(bucket)
// {
//   ... read memory/file buckets ...
// }
//
// Cleanup.
// mp4_process_context_exit(context);
//

MP4_DLL_EXPORT mp4_process_context_t*
mp4_process_context_init(mp4_global_context_t const* global_context);

MP4_DLL_EXPORT void
mp4_process_context_exit(mp4_process_context_t const* context);

MP4_DLL_EXPORT void
mp4_process_context_set_verbose(mp4_process_context_t* context,
                                fmp4_log_level_t level);

MP4_DLL_EXPORT headers_t*
mp4_process_context_get_headers(mp4_process_context_t const* context);

MP4_DLL_EXPORT buckets_t*
mp4_process_context_get_buckets(mp4_process_context_t const* context);

MP4_DLL_EXPORT mp4_options_t*
mp4_process_context_get_options(mp4_process_context_t const* context);

MP4_DLL_EXPORT fmp4_result
mp4_process_context_get_result(mp4_process_context_t const* context);

MP4_DLL_EXPORT char const*
mp4_process_context_get_result_text(mp4_process_context_t const* context);

MP4_DLL_EXPORT void
mp4_process_context_set_enable_mmap(mp4_process_context_t* context,
                                    uint32_t enable_mmap);

MP4_DLL_EXPORT void
mp4_process_context_set_download_callback(mp4_process_context_t* context,
                                          download_callback_t callback,
                                          void* callback_context);

MP4_DLL_EXPORT void
mp4_process_context_set_variable_callback(
  mp4_process_context_t* context, get_server_variable_callback_t callback,
  void* callback_context);

MP4_DLL_EXPORT void
mp4_process_context_set_log_error_callback(mp4_process_context_t* context,
                                           log_error_callback_t callback,
                                           void* callback_context);

MP4_DLL_EXPORT void
mp4_process_context_set_prefer_static(mp4_process_context_t* context,
                                      uint32_t prefer_static);

MP4_DLL_EXPORT void
mp4_process_context_set_iss_pass_through(mp4_process_context_t* context,
                                         uint32_t iss_pass_through);

MP4_DLL_EXPORT void
mp4_process_context_set_dynamic_time_shift_buffer_depth(
  mp4_process_context_t* context, uint32_t dynamic_time_shift_buffer_depth);

MP4_DLL_EXPORT void
mp4_process_context_set_is_storage_mpd(mp4_process_context_t* context,
                                       uint32_t is_storage_mpd);

MP4_DLL_EXPORT void
mp4_process_context_set_s3_parameters(mp4_process_context_t* context,
                                      char const* secret_key,
                                      char const* access_key,
                                      char const* region,
                                      char const* security_token,
                                      int use_headers);

MP4_DLL_EXPORT void
mp4_process_context_set_transcode_proxy_pass(mp4_process_context_t* context,
                                             char const* transcode_proxy_pass);

MP4_DLL_EXPORT void
mp4_process_context_set_transcoders_file(mp4_process_context_t* context,
                                         char const* transcoders_file);

MP4_DLL_EXPORT char const*
mp4_process_context_get_output_filter(mp4_process_context_t const* context);

MP4_DLL_EXPORT void
mp4_process_context_set_output_filter(mp4_process_context_t* context,
                                      char const* output_filter);

MP4_DLL_EXPORT void
mp4_process_context_set_content_id(mp4_process_context_t* context,
                                   int content_id);

// the entry point for all 'streaming' functions. I.e. anything that takes
// a virtual-file part.
MP4_NODISCARD MP4_DLL_EXPORT int
mp4_process(mp4_process_context_t* context, fmp4_http_method_t method,
            char const* url);

#ifdef __cplusplus
} /* extern C definitions */
#endif

#endif // MP4_PROCESS_H_AKW

// End Of File

